unit MMDirectoryView;

interface

uses
  Windows, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, ShellApi, FileCtrl, menus, ShlObj, ComObj, ActiveX, IniFiles;

type
  TIconSize = (isSmall, isLarge);

type
  TSystemDirectory = (SD_NO             ,
                      SD_DESKTOP        ,
                      SD_IE             ,
                      SD_SMPROGRAMS     ,
                      SD_SETTINGS       ,
                      SD_PRINTER        ,
                      SD_PERSONAL       ,
                      SD_FAVORITES      ,
                      SD_RUN            ,
                      SD_RECENT         ,
                      SD_SENDTO         ,
                      SD_PAPERBASKED    ,
                      SD_STARTMENU      ,
                      SD_MUSIC          ,
                      SD_VIDEOS         ,
                      SD_COMPUTER       ,
                      SD_NETWORK        ,
                      SD_FONTS          ,
                      SD_SHELLNEW       ,
                      SD_AUDESKTOP      ,
                      SD_APPDATA        ,
                      SD_PRINTHOOD      ,
                      SD_TEMPINETFILES  ,
                      SD_COOKIES        ,
                      SD_COURSE         ,
                      SD_AUAPPDATA      ,
                      SD_WINDOWS        ,
                      SD_SYSTEM         ,
                      SD_PROGRAMS       ,
                      SD_PICTURES       ,
                      SD_AUFILES        );

type TDirectoryChangingEvent = procedure(Sender: TObject; Directory: String; Node: TTreeNode; var AllowChange: Boolean) of object;
type TDirectoryAddingEvent = procedure(Sender: TObject; Directory: String; Empty, DirLink: Boolean; var AllowAdd: Boolean) of object;

type
  TMMDirectoryView = class(TTreeView)
  private
    FIconList: TImageList;
    FTaskList: TPopupMenu;
    FAllowDeleteDirectory: Boolean;
    FAllowCreateDirectory: Boolean;
    FAllowOpenDirectory: Boolean;
    FAllowRefresh: Boolean;
    FAllowDirChanging: Boolean;
    FAllowDirExpand: Boolean;
    FCurDirectory: String;
    FExpandingSelect: Boolean;
    FLoadImagesByCall: Boolean;
    FSelectOpenedDirs: Boolean;
    FDirectoriesToFind: Integer;
    FIconSize: TIconSize;
    FStartDirectory: TSystemDirectory;
    FDirectoryChanging: TDirectoryChangingEvent;
    FDirectoryAdding: TDirectoryAddingEvent;
    procedure TaskListPopup(Sender: TObject);
    procedure DeleteDirectory(Sender: TObject);
    procedure CreateDirectory(Sender: TObject);
    procedure OpenDirectory(Sender: TObject);
    procedure RefreshView(Sender: TObject);
    procedure FExpanding(Sender: TObject; Node: TTreeNode; var AllowExpansion: Boolean);
    procedure FCollapsing(Sender: TObject; Node: TTreeNode; var AllowCollaps: Boolean);
    procedure FChanging(Sender: TObject; Node: TTreeNode; var AllowChange: Boolean);
    procedure FGetSelectedIndex(Sender: TObject; Node: TTreeNode);
    procedure FGetImageIndex(Sender: TObject; Node: TTreeNode);
    function GetFDirectory: String;
    function GetFileIconHandle(FileName: String; Opened: Boolean = False): hIcon;
    function IsEmpty(Directory: String): Boolean;
    procedure SetAllowDeleteDirectory(Value: Boolean);
    procedure SetAllowCreateDirectory(Value: Boolean);
    procedure SetAllowOpenDirectory(Value: Boolean);
    procedure SetAllowRefresh(Value: Boolean);
    procedure SetIconSize(Value: TIconSize);
    procedure SetStartDirectory(Value: TSystemDirectory);
    procedure SetDirectoriesToFind(Value: Integer);
  protected
    procedure Loaded; override;
  public
    property Directory: String read GetFDirectory;
    function GetDirectory(Node: TTreeNode; NoDirLink: Boolean = False): String;
    function SetDirectory(Directory: String): TTreeNode;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property AllowDeleteDirectory: Boolean read FAllowDeleteDirectory write SetAllowDeleteDirectory;
    property AllowCreateDirectory: Boolean read FAllowCreateDirectory write SetAllowCreateDirectory;
    property AllowOpenDirectory: Boolean read FAllowOpenDirectory write SetAllowOpenDirectory;
    property AllowRefresh: Boolean read FAllowRefresh write SetAllowRefresh;
    property ExpandingSelect: Boolean read FExpandingSelect write FExpandingSelect;
    property SelectOpenedDirs: Boolean read FSelectOpenedDirs write FSelectOpenedDirs;
    property IconSize: TIconSize read FIconSize write SetIconSize;
    property LoadImagesByCall: Boolean read FLoadImagesByCall write FLoadImagesByCall;
    property StartDirectory: TSystemDirectory read FStartDirectory write SetStartDirectory;
    property DirectoriesToFind: Integer read FDirectoriesToFind write SetDirectoriesToFind;
    property OnDirectoryChanging: TDirectoryChangingEvent read FDirectoryChanging write FDirectoryChanging;
    property OnDirectoryAdding: TDirectoryAddingEvent read FDirectoryAdding write FDirectoryAdding;
    procedure UpdateDirectories(ChangeIconSize: TIconSize; ShowDir: String = '');
    procedure CreateDirectoryIn(Place: String);
    procedure DeleteThisDirectory(Directory: String);
  end;

var
  RightClickSelected: String;
  UseCurPath: Boolean = False;
  InitializeChanges: Boolean = True;
  CurPathExists: Boolean;
  dl: TStrings;
  CurPath: String;
  ErrorMode: Word;

const
  NotAllowed = ['\', '/', ':', '*', '?', '"', '<', '>', '|'];

procedure Register;
procedure GetDrives(var List: TStrings);
function IsDirLink(Dir: String): String;
function GetFileDisplayName(FileName: String): String;
function ShellFileOperation(Operation: Integer; WindowHandle: HWND; FromData: array of String; Confirm: Boolean; ToData: String = ''): Boolean;
function GetDrive(DriveDisplayName: String): String;
function GetFolder(FolderID: TSystemDirectory): String;
function GetLinkFileName(Link: String): String;
function AllowFileName(FileName: String): Boolean;

implementation

procedure Register;
begin
RegisterComponents('MM', [TMMDirectoryView]);
end;

constructor TMMDirectoryView.Create(AOwner: TComponent);
var mi: TMenuItem; i: Integer;
begin
inherited Create(AOwner);
ReadOnly := True;
RightClickSelect := True;
FExpandingSelect := True;
FSelectOpenedDirs := False;
FAllowDeleteDirectory := False;
FAllowCreateDirectory := True;
FAllowOpenDirectory := True;
FAllowRefresh := True;
FAllowDirExpand := True;
FIconSize := isSmall;
FLoadImagesByCall := True;
FDirectoriesToFind := faAnyFile;
FStartDirectory := SD_WINDOWS;
FTaskList := TPopupMenu.Create(Self);
mi := TMenuItem.Create(FTaskList);
mi.Name := 'New';
mi.Caption := 'Neuer Ord&ner';
mi.Visible := True;
mi.OnClick := CreateDirectory;
FTaskList.Items.Add(mi);
mi := TMenuItem.Create(FTaskList);
mi.Name := 'Delete';
mi.Caption := 'Ordner &lschen';
mi.Visible := False;
mi.OnClick := DeleteDirectory;
FTaskList.Items.Add(mi);
mi := TMenuItem.Create(FTaskList);
mi.Name := 'Open';
mi.Caption := 'Ordner im Ex&plorer ffnen';
mi.Visible := True;
mi.OnClick := OpenDirectory;
FTaskList.Items.Add(mi);
mi := TMenuItem.Create(FTaskList);
mi.Name := 'Refresh';
mi.Caption := '&Aktualisieren';
mi.Visible := True;
mi.OnClick := RefreshView;
FTaskList.Items.Add(mi);
FTaskList.OnPopup := TaskListPopup;
PopupMenu := FTaskList;
If csDesigning in ComponentState Then
  Exit;
FIconList := TImageList.Create(Self);
Images := FIconList;
end;

procedure TMMDirectoryView.Loaded;
var i: Integer; tn: TTreeNode; icon: TIcon; AllowAdd: Boolean;
begin
inherited Loaded;
ErrorMode := SetErrorMode(SEM_FailCriticalErrors);
If not (csDesigning in ComponentState) Then
 begin
  FIconList.BkColor := Color;
  If FIconSize = isLarge Then
   begin
    FIconList.Width := 32;
    FIconList.Height := 32;
   end;
  Items.Clear;
  dl := TStringList.Create;
  GetDrives(dl);
  FAllowDirChanging := False;
  Items.BeginUpdate;
  for i := 0 to dl.Count-1 do
   begin
    AllowAdd := True;
    If Assigned(FDirectoryAdding) Then
      OnDirectoryAdding(Self, dl[i], IsEmpty(dl[i]), False, AllowAdd);
    If AllowAdd Then
     begin
      tn := Items.Add(nil, GetFileDisplayName(dl[i]));
      icon := TIcon.Create;
      icon.Handle := GetFileIconHandle(dl[i]);
      tn.ImageIndex := FIconList.AddIcon(icon);
      icon.Free;
      If not IsEmpty(dl[i]) Then
        Items.AddChild(tn, '');
     end;
   end;
  Items.EndUpdate;
  OnExpanding := FExpanding;
  OnCollapsing := FCollapsing;
  OnChanging := FChanging;
  OnGetSelectedIndex := FGetSelectedIndex;
  If FLoadImagesByCall Then
    OnGetImageIndex := FGetImageIndex;
  InitializeChanges := False;
  If FStartDirectory <> SD_NO Then
    SetDirectory(GetFolder(FStartDirectory))
   else
   begin
    If AutoExpand Then
     begin
      tn := Items[0];
      while tn.getNextSibling <> nil do
       begin
        tn := tn.getNextSibling;
        If GetDriveType(PChar(GetDrive(tn.Text))) = DRIVE_FIXED Then
          Break;
       end;
      Selected := tn;
     end;
   end;
  FAllowDirChanging := True;
 end;
end;

destructor TMMDirectoryView.Destroy;
begin
dl.Free;
FAllowDirChanging := False;
SetErrorMode(ErrorMode);
inherited Destroy;
end;

function GetLinkFileName(Link: String): String;
begin
If AnsiUpperCase(ExtractFileExt(Link)) <> '.LNK' Then
  Result := Link+'.lnk'
 else
  Result := Link;
end;

procedure TMMDirectoryView.FGetImageIndex(Sender: TObject; Node: TTreeNode);
var s: String; icon: TIcon;
begin
If Node.ImageIndex = 0 Then
 begin
  icon := TIcon.Create;
  s := GetDirectory(Node, True);
  If AnsiUpperCase(ExtractFileExt(Copy(s, 1, Length(s)-1))) = '.LNK' Then
    s := Copy(s, 1, Length(s)-1);
  icon.Handle := GetFileIconHandle(s);
  Node.ImageIndex := FIconList.AddIcon(icon);
  icon.Free;
 end;
end;

function IsDirLink(Dir: String): String;
procedure GetLinkInformation(LinkFile: String; out Path, Parameters, Description, WorkingDirectory, IconFile: String; out IconPosition: Integer);
var swLinkFile: WideString; aShellLink: IShellLink; aPersistFile: IPersistFile; szData: array[0 .. MAX_PATH] of Char; aFindData: TWin32FindData; i: Integer;
begin
swLinkFile := LinkFile;
aShellLink := CreateComObject(CLSID_ShellLink) as IShellLink;
aPersistFile := aShellLink as IPersistFile;
OleCheck(aPersistFile.Load(PWideChar(swLinkFile), STGM_READ));
OleCheck(aShellLink.GetPath(szData, MAX_PATH, aFindData, SLGP_UNCPRIORITY));
Path := szData;
OleCheck(aShellLink.GetArguments(szData, MAX_PATH));
Parameters := szData;
OleCheck(aShellLink.GetDescription(szData, MAX_PATH));
Description := szData;
OleCheck(aShellLink.GetWorkingDirectory(szData, MAX_PATH));
WorkingDirectory := szData;
OleCheck(aShellLink.GetIconLocation(szData, MAX_PATH, i));
IconFile := szData;
IconPosition := i;
end;
var ini: TIniFile; s, s2: String; li: array[0..4] of String; i: Integer;
begin
Result := '';
If Length(Dir) <= 3 Then
  Exit;
s := Dir;
s2 := AnsiUpperCase(ExtractFileExt(s));
If s2 = '' Then
 begin
  If FileGetAttr(Dir) and faReadOnly = 0 Then
    Exit;
  If s[Length(s)] <> '\' Then
    s := s+'\';
  If not (FileExists(s+'Desktop.ini') and FileExists(s+'target.lnk')) Then
    Exit;
  ini := TIniFile.Create(s+'Desktop.ini');
  If not ((ini.ReadString('.ShellClassInfo', 'CLSID2', '') = '{0AFACED1-E828-11D1-9187-B532F1E9575D}') and (ini.ReadInteger('.ShellClassInfo', 'Flags', 0) = 2)) Then
    Exit;
  ini.Free;
  GetLinkInformation(s+'target.lnk', li[0], li[1], li[2], li[3], li[4], i);
  s := li[0];
  If (Length(s) = 0) or (s[Length(s)] <> '\') Then
    s := s+'\';
  If (not DirectoryExists(s)) or (s = '\') Then
    Exit;
  end
  else if s2 = '.LNK' Then
  begin
   If not FileExists(s) Then
     Exit;
   GetLinkInformation(s, li[0], li[1], li[2], li[3], li[4], i);
   s := li[0];
   If (GetDriveType(PChar(s)) <> DRIVE_REMOVABLE) and (GetDriveType(PChar(s)) <> DRIVE_CDROM) Then
    begin
     If not DirectoryExists(s) Then
       Exit;
    end;
   If s[Length(s)] <> '\' Then
     s := s+'\';
  end
  else
   Exit;
Result := s;
end;

function GetFolder(FolderID: TSystemDirectory): String;
var pidl: PItemIDList; Path: array[0..MAX_PATH] of Char; i: Integer;
begin
case FolderID of
  SD_NO             : Exit;
  SD_DESKTOP        : i := $00;
  SD_IE             : i := $01;
  SD_SMPROGRAMS     : i := $02;
  SD_SETTINGS       : i := $03;
  SD_PRINTER        : i := $04;
  SD_PERSONAL       : i := $05;
  SD_FAVORITES      : i := $06;
  SD_RUN            : i := $07;
  SD_RECENT         : i := $08;
  SD_SENDTO         : i := $09;
  SD_PAPERBASKED    : i := $0A;
  SD_STARTMENU      : i := $0B;
  SD_MUSIC          : i := $0C;
  SD_VIDEOS         : i := $0D;
  SD_COMPUTER       : i := $11;
  SD_NETWORK        : i := $12;
  SD_FONTS          : i := $14;
  SD_SHELLNEW       : i := $15;
  SD_AUDESKTOP      : i := $19;
  SD_APPDATA        : i := $1A;
  SD_PRINTHOOD      : i := $1B;
  SD_TEMPINETFILES  : i := $20;
  SD_COOKIES        : i := $21;
  SD_COURSE         : i := $22;
  SD_AUAPPDATA      : i := $23;
  SD_WINDOWS        : i := $24;
  SD_SYSTEM         : i := $25;
  SD_PROGRAMS       : i := $26;
  SD_PICTURES       : i := $27;
  SD_AUFILES        : i := $28;
 end;
If SUCCEEDED(SHGetSpecialFolderLocation(0, i, pidl)) Then
 begin
  SHGetPathFromIDList(pidl, Path);
  Result := Path;
 end;
end;

function ShellFileOperation(Operation: Integer; WindowHandle: HWND; FromData: array of String; Confirm: Boolean; ToData: String = ''): Boolean;
var i, i2: Integer; DataString: String; ShellInfo: TSHFileOpStructA; ks: TKeyboardState;
begin
DataString := '';
for i := Low(FromData) to High(FromData) do
  DataString := DataString+FromData[i]+#0;
DataString := DataString+#0;
i2 := Integer(Confirm)*FOF_CONFIRMMOUSE;
GetKeyboardState(ks);
If (ks[VK_SHIFT] and 128) = 0 Then
  Inc(i2, FOF_ALLOWUNDO);
with ShellInfo do
 begin
  wnd := WindowHandle;
  wFunc := Operation;
  pFrom := PChar(DataString);
  fFlags := i2;
  If Operation = FO_DELETE Then
    pTo := nil
   else
    pTo := PChar(ToData);
 end;
SHFileOperation(ShellInfo);
Result := not ShellInfo.fAnyOperationsAborted;
end;

procedure GetDrives(var List: TStrings);
var i: Integer; s: String;
begin
for i := 0 to 25 do
 begin
  s := Chr(65+i)+':\';
  If GetDriveType(PChar(s)) > 1 Then
  List.Add(s);
 end;
end;

function TMMDirectoryView.GetFileIconHandle(FileName: String; Opened: Boolean = False): HIcon;
var FileInfo: SHFileInfo; Attr: Cardinal;
begin
Attr := SHGFI_ICON;
If FIconSize = isSmall Then
  Attr := Attr or SHGFI_SMALLICON
 else
  Attr := Attr or SHGFI_LARGEICON;
If Opened Then
  Attr := Attr or SHGFI_OPENICON;
SHGetFileInfo(PChar(FileName), 0, FileInfo, SizeOf(FileInfo), Attr);
Result := FileInfo.hIcon;
end;

function GetFileDisplayName(FileName: String): String;
var FileInfo: SHFileInfo;
begin
SHGetFileInfo(PChar(FileName), 0, FileInfo, SizeOf(FileInfo), SHGFI_DISPLAYNAME);
Result := FileInfo.szDisplayName;
end;

function TMMDirectoryView.IsEmpty(Directory: String): Boolean;
var sr: TSearchRec; s, s2: String; dl: Boolean;
begin
s := Directory;
If (s[Length(s)] <> '\') and (AnsiUpperCase(ExtractFileExt(Directory)) <> '.LNK') Then
  s := s+'\';
Result := True;
{I-}
If ((Length(s) = 3) and ((GetDriveType(PChar(s)) = DRIVE_REMOVABLE) or (GetDriveType(PChar(s)) = DRIVE_CDROM))) or (FindFirst(s+'*', FDirectoriesToFind, sr) = 0) Then
  repeat
    s2 := IsDirLink(s+sr.Name);
    dl := s2 <> '';
    If not dl Then
      s2 := s+sr.Name;
    If ((((sr.Attr and faDirectory) > 0) or dl) and
       (sr.Name <> '.') and
       (sr.Name <> '..') and
       (GetFileDisplayName(s+sr.Name) <> '')) or
       ((Length(s) <= 3) and
       (((GetDriveType(PChar(s)) = DRIVE_REMOVABLE) or (GetDriveType(PChar(s)) = DRIVE_CDROM))) Then
      Result := False;
   until (not Result) or (FindNext(sr) <> 0);
{I+}
FindClose(sr);
end;

procedure TMMDirectoryView.FExpanding(Sender: TObject; Node: TTreeNode; var AllowExpansion: Boolean);
procedure SetIcon;
var icon: TIcon;
begin
If Node.ImageIndex = 0 Then
  FGetImageIndex(Sender, Node);
icon := TIcon.Create;
icon.Handle := GetFileIconHandle(GetDirectory(Node, True), True);
FIconList.ReplaceIcon(Node.ImageIndex, icon);
icon.Free;
end;
var sl: TStringList; sr: TSearchRec; i: Integer; tn: TTreeNode; ac, AllowAdd: Boolean; s: String; icon: TIcon;
begin
CurPath := GetDirectory(Node);
CurPathExists := DirectoryExists(CurPath);
UseCurPath := True;
If (csDestroying in ComponentState) or (not FAllowDirExpand) or ((not CurPathExists) and (not (((GetDriveType(PChar(CurPath)) = DRIVE_REMOVABLE) or (GetDriveType(PChar(CurPath)) = DRIVE_CDROM))))) Then
 begin
  UseCurPath := False;
  Exit;
 end;
If FExpandingSelect Then
  Selected := Node;
If Selected = Node Then
 begin
  ac := True;
  FChanging(nil, Node, ac);
 end;
If (Node.Count > 1) or ((Node.Count = 1) and (Node.Item[0].Text <> '')) or (csDesigning in ComponentState) Then
 begin
  UseCurPath := False;
  SetIcon;
  Exit;
 end;
sl := TStringList.Create;
If CurPathExists Then
 begin
  If FindFirst(CurPath+'*', FDirectoriesToFind, sr) = 0 Then
    repeat
      If (((sr.Attr and faDirectory) > 0) or (IsDirLink(CurPath+sr.Name) <> '')) and (sr.Name <> '.') and (sr.Name <> '..') and {(GetFileDisplayName(CurPath+sr.Name) <> '') and }AllowFileName(sr.Name) Then
       begin
        AllowAdd := True;
        If Assigned(FDirectoryAdding) Then
         begin
          s := IsDirLink(CurPath+sr.Name);
          If s = '' Then
            s := CurPath+sr.Name;
          OnDirectoryAdding(Self, CurPath+sr.Name+'\', IsEmpty({CurPath+sr.Name}s), IsDirLink(CurPath+sr.Name) <> '', AllowAdd);
         end;
        If AllowAdd Then
          sl.Add(sr.Name{GetFileDisplayName(CurPath+sr.Name)});
       end;
     until FindNext(sr) <> 0;
  FindClose(sr);
  sl.Sort;
 end;
Items.BeginUpdate;
If not ((sl.Count = 0) and ((GetDriveType(PChar(CurPath)) = DRIVE_REMOVABLE) or (GetDriveType(PChar(CurPath)) = DRIVE_CDROM))) Then
 begin
  Node.DeleteChildren;
  for i := 0 to sl.Count-1 do
   begin
    tn := Items.AddChild(Node, sl[i]);
    s := CurPath+sl[i];
    If (not DirectoryExists(s)) and (GetLinkFileName(s) <> s) and (IsDirLink(GetLinkFileName(s)) <> '') Then
      s := s+'.lnk';
    If not FLoadImagesByCall Then
     begin
      icon := TIcon.Create;
      icon.Handle := GetFileIconHandle(s);
      tn.ImageIndex := FIconList.AddIcon(icon);
      icon.Free;
     end;
    If IsDirLink(s) <> '' Then
      s := IsDirLink(s);
    If (not IsEmpty(s)) Then/
      Items.AddChild(tn, '');
   end;
  If Node.HasChildren Then
    SetIcon;
 end
 else
 begin
  If Node.Expanded Then
    Node.Collapse(False);
  AllowExpansion := False;
 end;
Items.EndUpdate;
sl.Free;
UseCurPath := False;
end;

function TMMDirectoryView.GetDirectory(Node: TTreeNode; NoDirLink: Boolean = False): String;
var s: String; n: TTreeNode; i, li: Integer;
begin
Result := '';
If Node = nil Then
  Exit;
n := Node;
If n.Level = 0 Then
  s := GetDrive(n.Text)
 else
  s := n.Text;
If s = '' Then
  Exit;
If s[Length(s)] <> '\' Then
  s := s+'\';
Result := s;
while n.Parent <> nil do
 begin
  n := n.Parent;
  If n.Level = 0 Then
    s := GetDrive(n.Text)
   else
    s := n.Text;
  If s[Length(s)] <> '\' Then
    s := s+'\';
  Result := s+Result;
 end;
If not (UseCurPath and (not CurPathExists)) Then
 begin
  i := 0;
  s := '';
  while i < Length(Result) do
   begin
    li := i+1;
    i := i+Pos('\', Copy(Result, i+1, Length(Result)));
    s := s+Copy(Result, li, i-li+1);
    If (Length(s) > 3) and (not DirectoryExists(s)) and ((IsDirLink(Copy(s, 1, Length(s)-1)+'.lnk') <> '') or (GetLinkFileName(Copy(s, 1, Length(s)-1)) = Copy(s, 1, Length(s)-1))) Then
     begin
      If (GetLinkFileName(Copy(s, 1, Length(s)-1)) = Copy(s, 1, Length(s)-1)) Then
        s := Copy(s, 1, Length(s)-1)
       else
        s := Copy(s, 1, Length(s)-1)+'.lnk';
     end;
    If (IsDirLink(s) <> '') and (not (NoDirLink and (i = Length(Result)))) Then
      s := IsDirLink(s);
   end;
  Result := s;
 end;
end;

function TMMDirectoryView.GetFDirectory: String;
begin
If Selected = nil Then
  Result := ''
 else
  Result := GetDirectory(Selected);
end;

function TMMDirectoryView.SetDirectory(Directory: String): TTreeNode;
var s, s2: String; i, i3: Integer; found, b, fdl: Boolean;
begin
If Directory = '' Then
  Exit;
s2 := Directory;
fdl := (not DirectoryExists(s2)) and (IsDirLink(s2) <> '');
If s2[Length(s2)] <> '\' Then
  s2 := s2+'\';
s := s2;
b := False;
repeat
  b := (FileGetAttr(s) or DirectoriesToFind > DirectoriesToFind);
  If Pos('\', s) = Length(s) Then
    s := ''
   else
   begin
    for i := Length(s)-1 downto 1 do
      If s[i] = '\' Then
        Break;
    s := Copy(s, 1, i);
   end;
 until b or (s = '');
If ((not DirectoryExists(Directory)) or b) and (not fdl) Then
 begin
  Result := nil;
  Exit;
 end;
FAllowDirChanging := False;
If AnsiUpperCase(s2) = AnsiUpperCase(GetDirectory(Selected, True)) Then
 begin
  FAllowDirChanging := True;
  Result := Selected;
  Exit;
 end;
s := AnsiUpperCase(Copy(s2, 1, 3));
for i := 0 to Items.Count do
  If (i = Items.Count) or (Items[i].Level = 0) and (Items[i].Text = GetFileDisplayName(s)) Then
    Break;
If i = Items.Count Then
 begin
  FAllowDirChanging := True;
  Result := nil;
  Exit;
 end;
Items.BeginUpdate;
If Length(s2) > 3 Then
 begin
  s2 := Copy(s2, 4, Length(s2));
  repeat
    Items[i].Expand(False);
    found := False;
    s := Copy(s2, 1, Pos('\', s2)-1);
    If Items[i].getFirstChild <> nil Then
     begin
      i3 := Items[i].getFirstChild.AbsoluteIndex;
      repeat
        If AnsiUpperCase(Items[i3].Text) = AnsiUpperCase(s) Then
         begin
          found := True;
          Break;
         end;
        If Items[i].GetNextChild(Items[i3]) <> nil Then
          i3 := Items[i].GetNextChild(Items[i3]).AbsoluteIndex;
       until  found or (Items[i].GetNextChild(Items[i3].getPrevSibling) = nil);
     end;
    s2 := Copy(s2, Pos('\', s2)+1, Length(s2));
    i := i3;
   until (not found) or (s2 = '');
 end
 else
  found := True;
If found Then
 begin
  FAllowDirChanging := True;
  Items[i].Expand(False);
  FChanging(Self, Selected, b);
  Selected := Items[i];
  Items[i].MakeVisible;
  Result := Items[i];
  b := True;
  //FDirectoryChanging(Self, GetDirectory(Result), Result, b);
 end
 else
  Result := nil;
FAllowDirChanging := True;
Items.EndUpdate;
end;

procedure TMMDirectoryView.UpdateDirectories(ChangeIconSize: TIconSize; ShowDir: String = '');
var i: Integer; tn: TTreeNode; icon: TIcon; AllowAdd: Boolean;
begin
FAllowDirChanging := False;
Items.BeginUpdate;
FAllowDirExpand := False;
Items.Clear;
FAllowDirExpand := True;
FIconList.Clear;
If ChangeIconSize <> FIconSize Then
 begin
  FIconSize := ChangeIconSize;
  If FIconSize = isSmall Then
    i := 16
   else
    i := 32;
  FIconList.Width := i;
  FIconList.Height := i;
  If i = 16 Then
    Indent := 19;
 end;
dl.Clear;
GetDrives(dl);
for i := 0 to dl.Count-1 do
 begin
  AllowAdd := True;
  If Assigned(FDirectoryAdding) Then
    OnDirectoryAdding(Self, dl[i], IsEmpty(dl[i]), False, AllowAdd);
  If AllowAdd Then
   begin
    tn := Items.Add(nil, GetFileDisplayName(dl[i]));
    icon := TIcon.Create;
    icon.Handle := GetFileIconHandle(dl[i]);
    tn.ImageIndex := FIconList.AddIcon(icon);
    icon.Free;
    If not IsEmpty(dl[i]) Then
      Items.AddChild(tn, '');
   end;
 end;
FAllowDirChanging := True;
Items.EndUpdate;
If DirectoryExists(ShowDir) Then
  SetDirectory(ShowDir);
end;

procedure TMMDirectoryView.TaskListPopup(Sender: TObject);
var exists: Boolean; s, s2: String;
begin
s := GetDirectory(Selected, True);
If (not DirectoryExists(s)) and (IsDirLink(s) <> '') Then
  s2 := IsDirLink(s)
 else
  s2 := s;
RightClickSelected := s;
exists := DirectoryExists(s2) or FileExists(s);
FTaskList.Items[0].Enabled := exists and DirectoryExists(s2);
FTaskList.Items[1].Enabled := exists and (Length(s) > 3);
end;

procedure TMMDirectoryView.SetAllowDeleteDirectory(Value: Boolean);
begin
FAllowDeleteDirectory := Value;
FTaskList.Items[1].Visible := FAllowDeleteDirectory;
end;

procedure TMMDirectoryView.SetAllowCreateDirectory(Value: Boolean);
begin
FAllowCreateDirectory := Value;
FTaskList.Items[0].Visible := FAllowCreateDirectory;
end;

procedure TMMDirectoryView.SetAllowRefresh(Value: Boolean);
begin
FAllowRefresh := Value;
FTaskList.Items[3].Visible := FAllowRefresh;
end;

procedure TMMDirectoryView.DeleteDirectory(Sender: TObject);
var s: String; newS: TTreeNode;
begin
s := RightClickSelected;
newS := SetDirectory(s).Parent;
If (not DirectoryExists(s)) and (IsDirLink(s) <> '') Then
  s := s+' ';
ShellFileOperation(FO_DELETE, Handle, [Copy(s, 1, Length(s)-1)], True);
If not DirectoryExists(s) Then
 begin
  OnCollapsing := nil;
  Selected := newS;
  OnCollapsing := FCollapsing;
  UpdateDirectories(FIconSize, GetFDirectory);
 end;
end;

procedure TMMDirectoryView.CreateDirectory(Sender: TObject);
var s, s2: String; first: Boolean;
begin
s := RightClickSelected;
If (not DirectoryExists(s)) and (IsDirLink(s) <> '') Then
  s := IsDirLink(s);
s2 := 'Neuer Ordner';
first := True;
repeat
  If not first Then
   begin
    MessageBeep(MB_ICONERROR);
    If MessageDlg('Das angegebene Verzeichnis existiert bereits. Mchten Sie einen anderen Namen angeben?', mtError, [mbYes, mbNo], 0) = mrNo Then
      Exit;
   end;
  If not InputQuery('Neuer Ordner', 'Geben Sie den Namen des neuen Ordners ein:', s2) Then
    Exit;
  first := False;
 until not DirectoryExists(s+s2);
If s2 = '' Then
 begin
  Application.MessageBox('Sie mssen einen Ordnernamen angeben.', 'Fehler', 48);
  Exit;
 end;
If ForceDirectories(s+s2) Then
  UpdateDirectories(FIconSize, s+s2)
 else
  Application.MessageBox(PChar('Der Ordner "'+s+s2+'" konnte nicht erstellt werden.'), 'Fehler', 16);
end;

procedure TMMDirectoryView.CreateDirectoryIn(Place: String);
var s: String;
begin
s := Place;
If s[Length(s)] <> '\' Then
  s := s+'\';
If DirectoryExists(Place) Then
 begin
  RightClickSelected := s;
  CreateDirectory(nil);
 end;
end;

procedure TMMDirectoryView.DeleteThisDirectory(Directory: String);
var s: String;
begin
s := Directory;
If s[Length(s)] <> '\' Then
  s := s+'\';
If DirectoryExists(s) and (Length(s) > 3) Then
 begin
  RightClickSelected := s;
  DeleteDirectory(nil);
 end;
end;

procedure TMMDirectoryView.FCollapsing(Sender: TObject; Node: TTreeNode; var AllowCollaps: Boolean);
var icon: TIcon; ac: Boolean;
begin
icon := TIcon.Create;
icon.Handle := GetFileIconHandle(GetDirectory(Node, True), False);
FIconList.ReplaceIcon(Node.ImageIndex, icon);
icon.Free;
If Selected = Node Then
 begin
  ac := True;
  FChanging(nil, Node, ac);
 end;
end;

procedure TMMDirectoryView.FChanging(Sender: TObject; Node: TTreeNode; var AllowChange: Boolean);
var s: String;
begin
If UseCurPath Then
  s := CurPath
 else
  s := GetDirectory(Node);
If Assigned(FDirectoryChanging) and (not (csDestroying in ComponentState)) and FAllowDirChanging and (s <> FCurDirectory) Then
 begin
  OnDirectoryChanging(Sender, s, Node, AllowChange);
  If not AllowChange Then
    Exit;
 end;
FCurDirectory := s;
end;

procedure TMMDirectoryView.SetIconSize(Value: TIconSize);
begin
If (csDesigning in ComponentState) or InitializeChanges Then
  FIconSize := Value;
end;

function GetDrive(DriveDisplayName: String): String;
var i: Integer;
begin
for i := 0 to dl.Count-1 do
  If Pos(Copy(dl[i], 1, 2), DriveDisplayName) > 0 Then
    Break;
Result := dl[i];
end;

procedure TMMDirectoryView.SetStartDirectory(Value: TSystemDirectory);
begin
If ((csDesigning in ComponentState) or (csLoading in ComponentState)) or (Value = SD_NO) Then
  FStartDirectory := Value;
end;

procedure TMMDirectoryView.SetDirectoriesToFind(Value: Integer);
begin
If Value < 64 Then
 begin
  FDirectoriesToFind := Value;
  If (not (csLoading in ComponentState)) and (not (csDesigning in ComponentState)) Then
    UpdateDirectories(FIconSize, GetFDirectory);
 end;
end;

procedure TMMDirectoryView.SetAllowOpenDirectory(Value: Boolean);
begin
FAllowOpenDirectory := Value;
FTaskList.Items[2].Visible := FAllowOpenDirectory;
end;

procedure TMMDirectoryView.OpenDirectory(Sender: TObject);
var s: String;
begin
s := RightClickSelected;
If (not DirectoryExists(s)) and (IsDirLink(s) <> '') Then
  s := IsDirLink(s);
If FSelectOpenedDirs Then
  ShellExecute(Application.Handle, 'open', PChar(GetFolder(SD_WINDOWS)+'\explorer'), PChar('/E,/select,"'+Copy(s, 1, Length(s)-1)+'"'), nil, SW_NORMAL)
 else
  ShellExecute(Application.Handle, 'explore', PChar(s), nil, nil, SW_NORMAL);
end;

procedure TMMDirectoryView.FGetSelectedIndex(Sender: TObject; Node: TTreeNode);
begin
Node.SelectedIndex := Node.ImageIndex;
end;

procedure TMMDirectoryView.RefreshView(Sender: TObject);
var s: String;
begin
s := RightClickSelected;
If (not DirectoryExists(s)) and (IsDirLink(s) <> '') Then
  s := IsDirLink(s);
UpdateDirectories(FIconSize, s);
end;

function AllowFileName(FileName: String): Boolean;
var i: Integer;
begin
Result := True;
for i := 1 to Length(FileName) do
  If FileName[i] in NotAllowed Then
   begin
    Result := False;
    Exit;
   end;
end;

end.
